/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: Ecat_CoE_Structs.h $

    $GlobalRev: 63951 $
    $Version: V2.5.42.0 $
    $Packaged: 2016-09-12 15:21 $

   Targets:
    rcX          : yes

   Description:

    EtherCAT CoE protocol structures

   Changes:

     Version    Date        Author   Description
     ----------------------------------------------------------------------------------
     29        22.02.2011  eo
        Usage of __PACKED_XXX defines replaced by __TLR_PACKED_XXX defines
        See issue #0012627: Ecat_CoE_Structs.h: Header uses __PACKED_XXX instead of __TLR_PACKED_XXX

     28        15.01.2008  sb
        list type is a part of the index list and always has to be the first field in a complete set of OdListResponse messages

     27        03.07.2007  sb
        added SDOINFO Error Response

     26        23.03.2007  sb
        Updated source header

      3        26.07.2005  SB
        Adjusted ECAT_COE_SDOINFO_GETOBJDESCRES_T according to
        Beckhoff reply

      2        27.06.2005  SB
        Added SDO INFO structures

      1        15.06.2005  SB       1.000
        Created

**************************************************************************************/
#ifndef __ECAT_COE_STRUCTS_H
#define __ECAT_COE_STRUCTS_H

/* pragma pack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_PACK_1(ECAT_COE_STRUCTS_PUBLIC)
#endif


/* CANopen over EtherCAT services */

__TLR_PACKED_PRE struct ECAT_COE_HEADER_Ttag
{
  TLR_UINT16 uNumber : 9;
  TLR_UINT16 uReserved : 3;
  TLR_UINT16 uService : 4;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_HEADER_Ttag ECAT_COE_HEADER_T;

#define ECAT_COE_SERVICE_EMERGENCY    (0x01)
#define ECAT_COE_SERVICE_SDOREQUEST   (0x02)
#define ECAT_COE_SERVICE_SDORESPONSE  (0x03)
#define ECAT_COE_SERVICE_TXPDO        (0x04)
#define ECAT_COE_SERVICE_RXPDO        (0x05)
#define ECAT_COE_SERVICE_TXPDOREMREQ  (0x06)
#define ECAT_COE_SERVICE_RXPDOREMREQ  (0x07)
#define ECAT_COE_SERVICE_SDOINFO      (0x08)

#define ECAT_COE_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - sizeof(ECAT_COE_HEADER_T))

__TLR_PACKED_PRE struct ECAT_MBX_COE_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  TLR_UINT8 bData[ECAT_COE_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_Ttag ECAT_MBX_COE_T;

INLINE TLR_UINT16 Ecs_GetCoEMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - sizeof(ECAT_COE_HEADER_T);
}

/******************************************************************************
 * CANopen over EtherCAT SDO packets                                          *
 ******************************************************************************/


/******************************************************************************
 * CoE SDOINIT structures
 */

#define ECAT_COE_SDOINITNORM_HEADER_SIZE (sizeof(ECAT_COE_HEADER_T) + sizeof(ECAT_COE_INITSDONORMHEADER_T))
#define ECAT_COE_SDOINITNORM_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOINITNORM_HEADER_SIZE)
#define ECAT_COE_SDOINITNORM_LEGACY_DATA_SIZE (0)

#define ECAT_COE_SDOINIT_HEADER_SIZE (sizeof(ECAT_COE_HEADER_T)+sizeof(ECAT_COE_INITSDOHEADER_T))
#define ECAT_COE_SDOINIT_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOINIT_HEADER_SIZE)
#define ECAT_COE_SDOINIT_LEGACY_DATA_SIZE (4)

__TLR_PACKED_PRE struct ECAT_COE_INITSDOHEADER_Ttag
{
  TLR_UINT8 uSizeIndicator    : 1;
  TLR_UINT8 uTransferType     : 1;
  TLR_UINT8 uDataSetSize      : 2;
  TLR_UINT8 uCompleteAccess   : 1;
  TLR_UINT8 uCommand          : 3;
  TLR_UINT16    usIndex;
  TLR_UINT8     ubSubIndex;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_INITSDOHEADER_Ttag ECAT_COE_INITSDOHEADER_T;

__TLR_PACKED_PRE struct ECAT_COE_INITSDONORMHEADER_Ttag
{
  TLR_UINT8 uSizeIndicator    : 1;
  TLR_UINT8 uTransferType     : 1;
  TLR_UINT8 uDataSetSize      : 2;
  TLR_UINT8 uCompleteAccess   : 1;
  TLR_UINT8 uCommand          : 3;
  TLR_UINT16    usIndex;
  TLR_UINT8     ubSubIndex;
  TLR_UINT32    uCompleteSize;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_INITSDONORMHEADER_Ttag ECAT_COE_INITSDONORMHEADER_T;

__TLR_PACKED_PRE struct ECAT_COE_SDOINIT_Ttag
{
  ECAT_COE_INITSDOHEADER_T tSdoHeader;
  TLR_UINT8 bData[ECAT_COE_SDOINIT_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOINIT_Ttag ECAT_COE_SDOINIT_T;

__TLR_PACKED_PRE struct ECAT_COE_SDOINITNORM_Ttag
{
  ECAT_COE_INITSDONORMHEADER_T tSdoHeader;
  TLR_UINT8 bData[ECAT_COE_SDOINITNORM_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOINITNORM_Ttag ECAT_COE_SDOINITNORM_T;

#define ECAT_COE_SDO_TRANSFER_NORMAL (0x00)
#define ECAT_COE_SDO_TRANSFER_EXPEDITED (0x01)

INLINE TLR_UINT16 Ecs_GetCoESdoInitNormMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_SDOINITNORM_HEADER_SIZE;
}

INLINE TLR_UINT16 Ecs_GetCoESdoInitMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_SDOINIT_HEADER_SIZE;
}

/******************************************************************************
 * CoE SDOSEG structures
 */

#define ECAT_COE_SDOSEG_HEADER_SIZE (sizeof(ECAT_COE_HEADER_T)+sizeof(ECAT_COE_SDOSEGHEADER_T))
#define ECAT_COE_SDOSEG_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOSEG_HEADER_SIZE)
#define ECAT_COE_SDOSEG_LEGACY_DATA_SIZE (7)

__TLR_PACKED_PRE struct ECAT_COE_SDOSEGHEADER_Ttag
{
  TLR_UINT8      fMoreFollows        : 1;
  TLR_UINT8      uSegDataSize        : 3;
  TLR_UINT8      fToggle             : 1;
  TLR_UINT8      uCommand            : 3;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOSEGHEADER_Ttag ECAT_COE_SDOSEGHEADER_T;

__TLR_PACKED_PRE struct ECAT_COE_SDOSEG_Ttag
{
  ECAT_COE_SDOSEGHEADER_T tHeader;
  TLR_UINT8 bData[ECAT_COE_SDOSEG_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOSEG_Ttag ECAT_COE_SDOSEG_T;

INLINE TLR_UINT16 Ecs_GetCoESdoSegMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_SDOSEG_HEADER_SIZE;
}

/******************************************************************************
 * CoE SDOABORT structures
 */

__TLR_PACKED_PRE struct ECAT_COE_SDOABORT_Ttag
{
  ECAT_COE_INITSDOHEADER_T tSdoHeader;
  TLR_UINT32 uAbortCode;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOABORT_Ttag ECAT_COE_SDOABORT_T;

#define ECAT_COE_SDOABORT_SIZE (sizeof(ECAT_COE_HEADER_T) + sizeof(ECAT_COE_SDOABORT_T))

#define ECAT_COE_SDO_ABORTCODE_TOGGLE_BIT_NOT_ALTERNATED          0x05030000

#define ECAT_COE_SDO_ABORTCODE_PROTOCOL_TIMEOUT                   0x05040000
#define ECAT_COE_SDO_ABORTCODE_COMMAND_SPECIFIER_NOT_VALID        0x05040001
#define	ECAT_COE_SDO_ABORTCODE_BLK_SIZE                           0x05040002	// Invalid block size (block mode only).
#define	ECAT_COE_SDO_ABORTCODE_SEQNO		                          0x05040003	// Invalid sequence number (block mode only).
#define	ECAT_COE_SDO_ABORTCODE_CRC                                0x05040004	// CRC error (block mode only).
#define ECAT_COE_SDO_ABORTCODE_OUT_OF_MEMORY                      0x05040005

#define ECAT_COE_SDO_ABORTCODE_UNSUPPORTED_ACCESS                 0x06010000
#define ECAT_COE_SDO_ABORTCODE_OBJECT_IS_WRITE_ONLY               0x06010001
#define ECAT_COE_SDO_ABORTCODE_OBJECT_IS_READ_ONLY                0x06010002
#define ECAT_COE_SDO_ABORTCODE_SUBINDEX_CANNOT_BE_WRITTEN_SI0_NZ  0x06010003
#define ECAT_COE_SDO_ABORTCODE_COMPLETE_ACCESS_NOT_SUPPORTED      0x06010004
#define ECAT_COE_SDO_ABORTCODE_OBJECT_LENGTH_EXCEEDS_MAILBOX_SIZE 0x06010005
#define ECAT_COE_SDO_ABORTCODE_OBJECT_MAPPED_TO_RXPDO_NO_WRITE    0x06010006
#define ECAT_COE_SDO_ABORTCODE_OBJECT_DOES_NOT_EXIST              0x06020000

#define ECAT_COE_SDO_ABORTCODE_OBJECT_CANNOT_BE_PDO_MAPPED        0x06040041
#define ECAT_COE_SDO_ABORTCODE_PDO_LENGTH_WOULD_EXCEED            0x06040042
#define ECAT_COE_SDO_ABORTCODE_GEN_PARAM_INCOMPATIBILITY          0x06040043
#define ECAT_COE_SDO_ABORTCODE_GEN_INTERNAL_INCOMPAT              0x06040047

#define ECAT_COE_SDO_ABORTCODE_ACCESS_FAILED_DUE_TO_HW_ERROR      0x06060000

#define ECAT_COE_SDO_ABORTCODE_DATATYPE_DOES_NOT_MATCH            0x06070010
#define ECAT_COE_SDO_ABORTCODE_DATATYPE_LENGTH_TOO_LONG           0x06070012
#define ECAT_COE_SDO_ABORTCODE_DATATYPE_LENGTH_TOO_SHORT          0x06070013

#define ECAT_COE_SDO_ABORTCODE_SUBINDEX_DOES_NOT_EXIST            0x06090011
#define ECAT_COE_SDO_ABORTCODE_RANGE_OF_PARAMETER_EXCEEDED        0x06090030
#define ECAT_COE_SDO_ABORTCODE_VALUE_OF_PARAM_WRITTEN_TO_HIGH     0x06090031
#define ECAT_COE_SDO_ABORTCODE_VALUE_OF_PARAM_WRITTEN_TO_LOW      0x06090032
#define ECAT_COE_SDO_ABORTCODE_MIN_VALUE_IS_LESS_THAN_MAX_VALUE   0x06090036

#define ECAT_COE_SDO_ABORTCODE_GENERAL_ERROR                      0x08000000
#define ECAT_COE_SDO_ABORTCODE_NO_TRANSFER_TO_APP                 0x08000020
#define ECAT_COE_SDO_ABORTCODE_LOCAL_CONTROL                      0x08000021
#define ECAT_COE_SDO_ABORTCODE_NO_TRANSFER_DUE_TO_CURRENT_STATE   0x08000022
#define ECAT_COE_SDO_ABORTCODE_NO_OBJECT_DICTIONARY_PRESENT       0x08000023

/******************************************************************************
 * CoE Emergency structures
 */

#define ECAT_COE_EMERGENCY_HEADER_SIZE (sizeof(ECAT_COE_HEADER_T) + sizeof(ECAT_COE_EMERGENCYHEADER_T))
#define ECAT_COE_EMERGENCY_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_EMERGENCY_HEADER_SIZE)

__TLR_PACKED_PRE struct ECAT_COE_EMERGENCYHEADER_Ttag
{
  TLR_UINT16    usErrorCode;
  TLR_UINT8     bErrorRegister;
  TLR_UINT8     abDiagnosticData[5];
} __TLR_PACKED_POST;

typedef struct ECAT_COE_EMERGENCYHEADER_Ttag ECAT_COE_EMERGENCYHEADER_T;

__TLR_PACKED_PRE struct ECAT_COE_EMERGENCY_Ttag
{
  ECAT_COE_EMERGENCYHEADER_T tEmergHdr;
  TLR_UINT8       abReserved[ECAT_COE_EMERGENCY_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_COE_EMERGENCY_Ttag ECAT_COE_EMERGENCY_T;

INLINE TLR_UINT16 Ecs_GetCoEEmergencyMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_EMERGENCY_HEADER_SIZE;
}

/******************************************************************************
 * CoE PDO structures
 */

#define ECAT_COE_PDO_HEADER_SIZE (sizeof(ECAT_COE_HEADER_T))
#define ECAT_COE_PDO_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_PDO_HEADER_SIZE)

INLINE TLR_UINT16 Ecs_GetCoEPdoMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_PDO_HEADER_SIZE;
}

/******************************************************************************
 * SDO info structures
 */

#define ECAT_COE_SDOINFO_HEADER_SIZE (sizeof(ECAT_COE_SDOINFO_HEADER_T) + sizeof(ECAT_COE_HEADER_T))
#define ECAT_COE_SDOINFO_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOINFO_HEADER_SIZE)

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_HEADER_Ttag
{
  TLR_UINT8     uOpCode : 7;
  TLR_UINT8     fIncomplete : 1;
  TLR_UINT8     uReserved : 8;
  TLR_UINT16    usFragmentsLeft;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOINFO_HEADER_Ttag ECAT_COE_SDOINFO_HEADER_T;

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_GETODLISTREQ_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT16                usListType;
} __TLR_PACKED_POST;

#define ECAT_COE_SDOINFO_GETODLISTRES_HEADER_SIZE (6)
#define ECAT_COE_SDOINFO_GETODLISTRES_MAX_NUM_ENTRIES ((ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOINFO_GETODLISTRES_HEADER_SIZE)/2)

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_GETODLISTRES_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT16                usIndexList[ECAT_COE_SDOINFO_GETODLISTRES_MAX_NUM_ENTRIES];
} __TLR_PACKED_POST;

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_GETOBJDESCREQ_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT16                usIndex;
} __TLR_PACKED_POST;

#define ECAT_COE_SDOINFO_GETOBJDESCRES_HEADER_SIZE (12)
#define ECAT_COE_SDOINFO_GETOBJDESCRES_MAX_NAME_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOINFO_GETOBJDESCRES_HEADER_SIZE)

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_GETOBJDESCRES_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT16                usIndex;
  TLR_UINT16                usDataType;
  TLR_UINT8                 bMaxSubindex;
  TLR_UINT8                 bObjectType;
  TLR_STR                   szName[ECAT_COE_SDOINFO_GETOBJDESCRES_MAX_NAME_SIZE];
} __TLR_PACKED_POST;

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_GETENTRYDESCREQ_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT16                usIndex;
  TLR_UINT8                 bSubindex;
  TLR_UINT8                 bValueInfo;
} __TLR_PACKED_POST;

#define ECAT_COE_SDOINFO_GETENTRYDESCRES_HEADER_SIZE (16)
#define ECAT_COE_SDOINFO_GETENTRYDESCRES_MAX_DATA_SIZE (ECAT_MBX_MAX_DATA_SIZE - ECAT_COE_SDOINFO_GETENTRYDESCRES_HEADER_SIZE)
#define ECAT_COE_SDOINFO_GETENTRYDESCRES_RESPONSE_HEADER_SIZE (10)

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_GETENTRYDESCRES_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT16                usIndex;
  TLR_UINT8                 bSubindex;
  TLR_UINT8                 bValueInfo;
  TLR_UINT16                usDataType;
  TLR_UINT16                usBitLength;
  TLR_UINT16                usObjAccess;
  TLR_UINT8                 abData[ECAT_COE_SDOINFO_GETENTRYDESCRES_MAX_DATA_SIZE];
}  __TLR_PACKED_POST;

#define ECAT_COE_SDOINFO_ERROR_SIZE (10)

__TLR_PACKED_PRE struct ECAT_COE_SDOINFO_ERROR_Ttag
{
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
  TLR_UINT32                ulAbortCode;
} __TLR_PACKED_POST;

typedef struct ECAT_COE_SDOINFO_GETODLISTREQ_Ttag ECAT_COE_SDOINFO_GETODLISTREQ_T;
typedef struct ECAT_COE_SDOINFO_GETODLISTRES_Ttag ECAT_COE_SDOINFO_GETODLISTRES_T;
typedef struct ECAT_COE_SDOINFO_GETOBJDESCREQ_Ttag ECAT_COE_SDOINFO_GETOBJDESCREQ_T;
typedef struct ECAT_COE_SDOINFO_GETOBJDESCRES_Ttag ECAT_COE_SDOINFO_GETOBJDESCRES_T;
typedef struct ECAT_COE_SDOINFO_GETENTRYDESCREQ_Ttag ECAT_COE_SDOINFO_GETENTRYDESCREQ_T;
typedef struct ECAT_COE_SDOINFO_GETENTRYDESCRES_Ttag ECAT_COE_SDOINFO_GETENTRYDESCRES_T;
typedef struct ECAT_COE_SDOINFO_ERROR_Ttag ECAT_COE_SDOINFO_ERROR_T;

#define ECAT_COE_SDOINFO_GETODLISTREQ (0x01)
#define ECAT_COE_SDOINFO_GETODLISTRES (0x02)
#define ECAT_COE_SDOINFO_GETOBJDESCREQ (0x03)
#define ECAT_COE_SDOINFO_GETOBJDESCRES (0x04)
#define ECAT_COE_SDOINFO_GETENTRYDESCREQ (0x05)
#define ECAT_COE_SDOINFO_GETENTRYDESCRES (0x06)
#define ECAT_COE_SDOINFO_ERROR_RESPONSE (0x07)

#define ECAT_COE_OBJCODE_DOMAIN (0x02)
#define ECAT_COE_OBJCODE_DEFTYPE (0x05)
#define ECAT_COE_OBJCODE_DEFSTRUCT (0x06)
#define ECAT_COE_OBJCODE_VAR (0x07)
#define ECAT_COE_OBJCODE_ARRAY (0x08)
#define ECAT_COE_OBJCODE_RECORD (0x09)

#define ECAT_COE_OBJLIST_LENGTH (0x0000)
#define ECAT_COE_OBJLIST_ALL (0x0001)
#define ECAT_COE_OBJLIST_RXPDO_MAPPABLE (0x0002)
#define ECAT_COE_OBJLIST_TXPDO_MAPPABLE (0x0003)
#define ECAT_COE_OBJLIST_BACKUP         (0x0004)
#define ECAT_COE_OBJLIST_CONFIG_DATA (0x0005)

#define ECAT_COE_SDOINFO_VALUEINFO_ACCESS_RIGHTS (1<<0)
#define ECAT_COE_SDOINFO_VALUEINFO_OBJECT_CATEGORY (1<<1)
#define ECAT_COE_SDOINFO_VALUEINFO_PDO_MAPPABLE (1<<2)
#define ECAT_COE_SDOINFO_VALUEINFO_UNIT_TYPE (1<<3)
#define ECAT_COE_SDOINFO_VALUEINFO_DEFAULT_VALUE (1<<4)
#define ECAT_COE_SDOINFO_VALUEINFO_MINIMUM_VALUE (1<<5)
#define ECAT_COE_SDOINFO_VALUEINFO_MAXIMUM_VALUE (1<<6)

INLINE TLR_UINT16 Ecs_GetCoEOdListResMaxNumEntries(TLR_UINT16 usMbxSize)
{
  return (Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_SDOINFO_GETODLISTRES_HEADER_SIZE) / 2;
}

INLINE TLR_UINT16 Ecs_GetCoEObjDescResMaxNameSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_SDOINFO_GETOBJDESCRES_HEADER_SIZE;
}

INLINE TLR_UINT16 Ecs_GetCoEEntryDescResMaxDataSize(TLR_UINT16 usMbxSize)
{
  return Ecs_MbxGetMaxDataSize(usMbxSize) - ECAT_COE_SDOINFO_GETENTRYDESCRES_HEADER_SIZE;
}

/******************************************************************************
 * Mailbox embedded INITSDO messages
 */

 __TLR_PACKED_PRE struct ECAT_MBX_COE_SDOINITNORM_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  ECAT_COE_INITSDONORMHEADER_T tSdoHeader;
  TLR_UINT8 bData[ECAT_COE_SDOINITNORM_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_SDOINITNORM_Ttag ECAT_MBX_COE_SDOINITNORM_T;

__TLR_PACKED_PRE struct ECAT_MBX_COE_SDOINIT_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  ECAT_COE_INITSDOHEADER_T tSdoHeader;
  TLR_UINT8 bData[ECAT_COE_SDOINIT_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_SDOINIT_Ttag ECAT_MBX_COE_SDOINIT_T;

/******************************************************************************
 * Mailbox embedded SDOSEG messages
 */

__TLR_PACKED_PRE struct ECAT_MBX_COE_SDOSEG_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  ECAT_COE_SDOSEGHEADER_T tSdoHeader;
  TLR_UINT8 bData[ECAT_COE_SDOSEG_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_SDOSEG_Ttag ECAT_MBX_COE_SDOSEG_T;


/******************************************************************************
 * Mailbox embedded SDOABORT messages
 */

__TLR_PACKED_PRE struct ECAT_MBX_COE_SDOABORT_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  ECAT_COE_SDOABORT_T tSdoAbort;
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_SDOABORT_Ttag ECAT_MBX_COE_SDOABORT_T;

/******************************************************************************
 * Mailbox embedded EMERGENCY message
 */

__TLR_PACKED_PRE struct ECAT_MBX_COE_EMERGENCY_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  ECAT_COE_EMERGENCY_T tEmergency;
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_EMERGENCY_Ttag ECAT_MBX_COE_EMERGENCY_T;

/******************************************************************************
 * Mailbox embedded PDO message
 */

__TLR_PACKED_PRE struct ECAT_MBX_COE_PDO_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  TLR_UINT8 abData[ECAT_COE_PDO_MAX_DATA_SIZE];
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_PDO_Ttag ECAT_MBX_COE_PDO_T;

/******************************************************************************
 * Mailbox embedded SDO info messages
 */

#define ECAT_COE_EMBED_SDOINFO_STRUCT(x,structName) \
  __TLR_PACKED_PRE struct ECAT_MBX_COE_SDOINFO_##x##_Ttag \
  { \
    ECAT_MBXHEADER_T tMbxHeader; \
    ECAT_COE_HEADER_T tCoEHeader; \
    ECAT_COE_SDOINFO_##x##_T structName; \
  } __TLR_PACKED_POST; \
  typedef struct ECAT_MBX_COE_SDOINFO_##x##_Ttag ECAT_MBX_COE_SDOINFO_##x##_T

ECAT_COE_EMBED_SDOINFO_STRUCT(GETODLISTREQ,tGetOdListReq);
ECAT_COE_EMBED_SDOINFO_STRUCT(GETODLISTRES,tGetOdListRes);
ECAT_COE_EMBED_SDOINFO_STRUCT(GETOBJDESCREQ,tGetObjDescReq);
ECAT_COE_EMBED_SDOINFO_STRUCT(GETOBJDESCRES,tGetObjDescRes);
ECAT_COE_EMBED_SDOINFO_STRUCT(GETENTRYDESCREQ,tGetEntryDescReq);
ECAT_COE_EMBED_SDOINFO_STRUCT(GETENTRYDESCRES,tGetEntryDescRes);
ECAT_COE_EMBED_SDOINFO_STRUCT(ERROR,tErrorRes);

__TLR_PACKED_PRE struct ECAT_MBX_COE_SDOINFO_HEADER_Ttag
{
  ECAT_MBXHEADER_T tMbxHeader;
  ECAT_COE_HEADER_T tCoEHeader;
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
} __TLR_PACKED_POST;

typedef struct ECAT_MBX_COE_SDOINFO_HEADER_Ttag ECAT_MBX_COE_SDOINFO_HEADER_T;

/******************************************************************************
 * Union of structures for SDO info
 */

union ECAT_COE_SDOINFO_Ttag
{
  ECAT_COE_SDOINFO_GETODLISTREQ_T tGetOdListReq;
  ECAT_COE_SDOINFO_GETODLISTRES_T tGetOdListRes;
  ECAT_COE_SDOINFO_GETOBJDESCREQ_T tGetObjDescReq;
  ECAT_COE_SDOINFO_GETOBJDESCRES_T tGetObjDescRes;
  ECAT_COE_SDOINFO_GETENTRYDESCREQ_T tGetEntryDescReq;
  ECAT_COE_SDOINFO_GETENTRYDESCRES_T tGetEntryDescRes;
  ECAT_COE_SDOINFO_ERROR_T      tErrorRes;
  ECAT_COE_SDOINFO_HEADER_T tSdoInfoHeader;
};

typedef union ECAT_COE_SDOINFO_Ttag ECAT_COE_SDOINFO_T;

union ECAT_MBX_COE_SDOINFO_Ttag
{
  ECAT_MBX_COE_SDOINFO_GETODLISTREQ_T tGetOdListReq;
  ECAT_MBX_COE_SDOINFO_GETODLISTRES_T tGetOdListRes;
  ECAT_MBX_COE_SDOINFO_GETOBJDESCREQ_T tGetObjDescReq;
  ECAT_MBX_COE_SDOINFO_GETOBJDESCRES_T tGetObjDescRes;
  ECAT_MBX_COE_SDOINFO_GETENTRYDESCREQ_T tGetEntryDescReq;
  ECAT_MBX_COE_SDOINFO_GETENTRYDESCRES_T tGetEntryDescRes;
  ECAT_MBX_COE_SDOINFO_ERROR_T tErrorRes;
  ECAT_MBX_COE_SDOINFO_HEADER_T tSdoInfoHeader;
};

typedef union ECAT_MBX_COE_SDOINFO_Ttag ECAT_MBX_COE_SDOINFO_T;

/******************************************************************************
 * Unions of structures for SDO
 */

union ECAT_SDO_HEADER_Ttag
{
  ECAT_COE_SDOSEGHEADER_T tSdoSegHeader;
  ECAT_COE_INITSDOHEADER_T tSdoInitHeader;
  ECAT_COE_INITSDONORMHEADER_T tSdoInitNormHeader;
};

typedef union ECAT_SDO_HEADER_Ttag ECAT_SDO_HEADER_T;

union ECAT_COE_SDO_Ttag
{
  ECAT_COE_SDOSEG_T tSdoSeg;
  ECAT_COE_SDOINIT_T tSdoInit;
  ECAT_COE_SDOINITNORM_T tSdoInitNorm;
  ECAT_COE_SDOABORT_T tSdoAbort;
};

typedef union ECAT_COE_SDO_Ttag ECAT_COE_SDO_T;

union ECAT_MBX_COE_SDO_Ttag
{
  ECAT_MBX_COE_SDOSEG_T tSdoSeg;
  ECAT_MBX_COE_SDOINIT_T tSdoInit;
  ECAT_MBX_COE_SDOINITNORM_T tSdoInitNorm;
  ECAT_MBX_COE_SDOABORT_T tSdoAbort;
};

typedef union ECAT_MBX_COE_SDO_Ttag ECAT_MBX_COE_SDO_T;


/******************************************************************************
 * SDO commands
 */

#define	ECAT_COE_SDO_INITIATEDOWNLOADREQ    0x01
#define	ECAT_COE_SDO_INITIATEDOWNLOADRES    0x03
#define	ECAT_COE_SDO_DOWNLOADSEGMENTREQ     0x00
#define	ECAT_COE_SDO_DOWNLOADSEGMENTRES     0x01
#define	ECAT_COE_SDO_INITIATEUPLOADREQ      0x02
#define	ECAT_COE_SDO_INITIATEUPLOADRES      0x02
#define	ECAT_COE_SDO_UPLOADSEGMENTREQ       0x03
#define	ECAT_COE_SDO_UPLOADSEGMENTRES       0x00
#define	ECAT_COE_SDO_ABORTTRANSFER          0x04


/* pragma unpack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_UNPACK_1(ECAT_COE_STRUCTS_PUBLIC)
#endif

#endif /* #ifndef __ECAT_COE_STRUCTS_H */
